<?php
/*
Plugin Name: بلاک هوشمند وردپرس
Description: مدیریت هوشمند اتصالات خارجی وردپرس با قابلیت تشخیص خودکار و امنیت پیشرفته در زمان قطعی نت بین الملل که باعث تسهیل در انجام امور روزانه مدیر سایت خواهد شد.
Version: 2.0.0
Author: مهندس براتی
*/

// جلوگیری از دسترسی مستقیم
if (!defined('ABSPATH')) {
    exit;
}

// تعریف ثابت‌های افزونه
define('WP_SMART_LIMIT_FILE', __FILE__);
define('WP_SMART_LIMIT_PLUGIN_BASENAME', plugin_basename(WP_SMART_LIMIT_FILE));
define('WP_SMART_LIMIT_VERSION', '2.0.0');

class WP_Smart_Limit {
    private static $instance = null;
    private $plugin_version;
    private $plugin_slug;
    private $cache_key = 'wp_smart_limit_blocked_hosts_cache';
    private $suspicious_threshold = 5;

    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        $this->plugin_slug = dirname(WP_SMART_LIMIT_PLUGIN_BASENAME);
        $this->plugin_version = WP_SMART_LIMIT_VERSION;

        add_action('admin_menu', [$this, 'admin_menu']);
        add_filter('pre_http_request', [$this, 'handle_http_request'], 10, 3);
        add_action('admin_init', [$this, 'admin_init']);
        add_action('wp_ajax_wp_smart_limit_quick_action', [$this, 'handle_quick_action']);
        add_action('wp_ajax_wp_smart_limit_load_tab', [$this, 'handle_load_tab']);
        add_action('admin_footer', [$this, 'show_new_domain_toast']);
        add_filter('plugin_action_links_' . WP_SMART_LIMIT_PLUGIN_BASENAME, [$this, 'add_plugin_action_links']);
        
        // فیلترهای اضافی برای بلاک کامل به‌روزرسانی‌ها
        add_filter('pre_site_transient_update_core', [$this, 'block_update_transients']);
        add_filter('pre_site_transient_update_plugins', [$this, 'block_update_transients']);
        add_filter('pre_site_transient_update_themes', [$this, 'block_update_transients']);
        add_filter('http_request_host_is_external', [$this, 'check_external_host'], 10, 3);
    }
    
    public function add_plugin_action_links($links) {
        $settings_link = '<a href="' . admin_url('options-general.php?page=smart-block') . '" style="font-weight: 700; color: #000;">تنظیمات</a>';
        array_unshift($links, $settings_link);
        return $links;
    }

    public function admin_menu() {
        add_submenu_page(
            'options-general.php',
            'بلاک هوشمند وردپرس',
            'بلاک هوشمند وردپرس',
            'manage_options',
            'smart-block',
            [$this, 'render_admin_page']
        );
    }

    public function admin_init() {}

    public function show_new_domain_toast() {
        $screen = get_current_screen();
        if ($screen && $screen->id !== 'settings_page_smart-block') {
            return;
        }

        $new_domains = get_transient('wp_smart_limit_new_domains');
        if ($new_domains && is_array($new_domains) && !empty($new_domains)) {
            ?>
            <script>
            jQuery(document).ready(function($) {
                var domains = <?php echo json_encode($new_domains); ?>;
                if (domains.length > 0) {
                    var message = 'دامنه جدید شناسایی شد: ' + domains.join(', ');
                    wpSmartLimitShowToast(message, 'info');
                }
            });
            </script>
            <?php
            delete_transient('wp_smart_limit_new_domains');
        }
    }

    public function render_admin_page() {
        if (!current_user_can('manage_options')) {
            return;
        }

        $tab = sanitize_key($_GET['tab'] ?? 'block');
        
        // پردازش فرم‌ها
        if ($tab == 'block' && $_SERVER['REQUEST_METHOD'] === 'POST') {
            $this->handle_form_submissions();
        }
        
        $logged_hosts = $this->get_logged_hosts();
        $blocked_hosts = get_option('wp_smart_limit_blocked_hosts', []);
        $suspicious_hosts = $this->detect_suspicious_hosts();
        ?>
        
        <div class="wrap wp-smart-limit-wrap">
            <style>
                /* حذف باکس سفید اضافی */
                .wp-smart-limit-wrap .notice,
                .wp-smart-limit-wrap .update-nag {
                    display: none !important;
                }
                
                .wp-smart-limit-wrap {
                    max-width: 1400px;
                    margin: 20px auto;
                    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen-Sans, Ubuntu, Cantarell, sans-serif;
                }
                
                /* تمام متن‌ها مشکی و بولد */
                .wp-smart-limit-wrap,
                .wp-smart-limit-wrap * {
                    color: #000 !important;
                    font-weight: 700 !important;
                }
                
                .wp-smart-limit-header {
                    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                    color: white !important;
                    padding: 30px;
                    border-radius: 12px;
                    margin-bottom: 30px;
                    box-shadow: 0 10px 25px rgba(102, 126, 234, 0.2);
                }
                
                .wp-smart-limit-header h1,
                .wp-smart-limit-header p {
                    color: white !important;
                    font-weight: 700 !important;
                }
                
                .wp-smart-limit-header h1 {
                    margin: 0 0 10px 0;
                    font-size: 32px;
                }
                
                .wp-smart-limit-header p {
                    margin: 0;
                    opacity: 0.95;
                    font-size: 16px;
                }
                
                .nav-tab-wrapper {
                    margin-bottom: 25px;
                    border-bottom: 2px solid #e1e4e8;
                }
                
                .nav-tab {
                    background: white;
                    border: 2px solid transparent;
                    color: #000 !important;
                    padding: 12px 25px;
                    margin: 0 5px 0 0;
                    border-radius: 8px 8px 0 0;
                    font-weight: 700 !important;
                    transition: all 0.3s;
                    border-bottom: none;
                    cursor: pointer;
                    text-decoration: none;
                }
                
                .nav-tab:hover {
                    color: #667eea !important;
                    background: #f8f9ff;
                }
                
                .nav-tab-active {
                    background: white;
                    color: #667eea !important;
                    border-color: #667eea;
                    border-bottom: 2px solid white;
                    margin-bottom: -2px;
                }
                
                .wp-smart-limit-stats {
                    display: grid;
                    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
                    gap: 20px;
                    margin-bottom: 30px;
                }
                
                .stat-card {
                    background: white;
                    padding: 25px;
                    border-radius: 10px;
                    box-shadow: 0 2px 10px rgba(0,0,0,0.08);
                    text-align: center;
                    transition: transform 0.2s, box-shadow 0.2s;
                }
                
                .stat-card:hover {
                    transform: translateY(-5px);
                    box-shadow: 0 5px 20px rgba(0,0,0,0.12);
                }
                
                .stat-number {
                    font-size: 36px;
                    font-weight: 700 !important;
                    color: #667eea !important;
                    margin: 0 0 8px 0;
                }
                
                .stat-label {
                    color: #000 !important;
                    font-size: 14px;
                    font-weight: 700 !important;
                }
                
                .wp-smart-limit-section {
                    background: white;
                    padding: 30px;
                    border-radius: 10px;
                    margin-bottom: 20px;
                    box-shadow: 0 2px 10px rgba(0,0,0,0.08);
                }
                
                .section-title {
                    font-size: 22px;
                    font-weight: 700 !important;
                    margin: 0 0 20px 0;
                    color: #000 !important;
                    border-bottom: 3px solid #667eea;
                    padding-bottom: 10px;
                    display: inline-block;
                }
                
                .add-host-form {
                    display: flex;
                    gap: 12px;
                    margin-bottom: 25px;
                    align-items: center;
                }
                
                .add-host-form input[type="text"] {
                    flex: 1;
                    padding: 12px 18px;
                    border: 2px solid #e1e4e8;
                    border-radius: 8px;
                    font-size: 15px;
                    font-weight: 700 !important;
                    color: #000 !important;
                    transition: border-color 0.2s, box-shadow 0.2s;
                }
                
                .add-host-form input[type="text"]:focus {
                    outline: none;
                    border-color: #667eea;
                    box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
                }
                
                /* دکمه‌ها با گرادینت پررنگ */
                .btn-primary {
                    background: linear-gradient(135deg, #4a5bc7 0%, #5a2d82 100%);
                    color: white !important;
                    border: none;
                    padding: 12px 30px;
                    border-radius: 8px;
                    font-weight: 700 !important;
                    cursor: pointer;
                    transition: transform 0.2s, box-shadow 0.2s;
                    font-size: 15px;
                }
                
                .btn-primary:hover {
                    transform: translateY(-2px);
                    box-shadow: 0 5px 15px rgba(74, 91, 199, 0.5);
                }
                
                .hosts-table {
                    width: 100%;
                    border-collapse: collapse;
                    margin-top: 20px;
                }
                
                .hosts-table thead {
                    background: linear-gradient(135deg, #4a5bc7 0%, #5a2d82 100%);
                }
                
                .hosts-table thead th {
                    color: white !important;
                    padding: 15px;
                    text-align: right;
                    font-weight: 700 !important;
                    font-size: 14px;
                }
                
                .hosts-table tbody tr {
                    border-bottom: 1px solid #f0f0f0;
                    transition: background-color 0.2s;
                }
                
                .hosts-table tbody tr:hover {
                    background-color: #f8f9ff;
                }
                
                .hosts-table tbody td {
                    padding: 15px;
                    text-align: right;
                    color: #000 !important;
                    font-weight: 700 !important;
                }
                
                .host-url {
                    font-family: 'Courier New', monospace;
                    color: #667eea !important;
                    font-weight: 700 !important;
                    direction: ltr;
                    text-align: left;
                }
                
                .host-count {
                    color: #666 !important;
                    font-size: 12px;
                    margin-right: 8px;
                }
                
                .badge {
                    display: inline-block;
                    padding: 5px 12px;
                    border-radius: 20px;
                    font-size: 12px;
                    font-weight: 700 !important;
                }
                
                .badge-blocked {
                    background: #fee;
                    color: #c33 !important;
                }
                
                .badge-active {
                    background: #e8f5e9;
                    color: #2e7d32 !important;
                }
                
                .badge-suspicious {
                    background: #fff3e0;
                    color: #ef6c00 !important;
                }
                
                .checkbox-wrapper {
                    display: flex;
                    align-items: center;
                    justify-content: center;
                }
                
                .checkbox-wrapper input[type="checkbox"] {
                    width: 20px;
                    height: 20px;
                    cursor: pointer;
                    accent-color: #667eea;
                }
                
                .quick-actions {
                    display: flex;
                    gap: 8px;
                    justify-content: center;
                }
                
                /* دکمه‌ها بدون ایکون و با گرادینت پررنگ */
                .quick-action-btn {
                    padding: 8px 16px;
                    border-radius: 6px;
                    border: none;
                    cursor: pointer;
                    font-size: 13px;
                    font-weight: 700 !important;
                    transition: all 0.2s;
                    color: white !important;
                }
                
                .btn-block {
                    background: linear-gradient(135deg, #d32f2f 0%, #b71c1c 100%);
                }
                
                .btn-block:hover {
                    background: linear-gradient(135deg, #f44336 0%, #d32f2f 100%);
                    transform: translateY(-2px);
                }
                
                .btn-unblock {
                    background: linear-gradient(135deg, #388e3c 0%, #2e7d32 100%);
                }
                
                .btn-unblock:hover {
                    background: linear-gradient(135deg, #4caf50 0%, #388e3c 100%);
                    transform: translateY(-2px);
                }
                
                .action-buttons {
                    margin-top: 25px;
                    padding-top: 20px;
                    border-top: 2px solid #f0f0f0;
                    display: flex;
                    gap: 12px;
                    flex-wrap: wrap;
                }
                
                .btn-save {
                    background: linear-gradient(135deg, #0d7a6e 0%, #2bb584 100%);
                    color: white !important;
                    border: none;
                    padding: 12px 35px;
                    border-radius: 8px;
                    font-weight: 700 !important;
                    cursor: pointer;
                    font-size: 15px;
                    transition: transform 0.2s, box-shadow 0.2s;
                }
                
                .btn-save:hover {
                    transform: translateY(-2px);
                    box-shadow: 0 8px 20px rgba(13, 122, 110, 0.5);
                }
                
                .btn-select-all {
                    background: linear-gradient(135deg, #d81b60 0%, #c2185b 100%);
                    color: white !important;
                    border: none;
                    padding: 12px 30px;
                    border-radius: 8px;
                    font-weight: 700 !important;
                    cursor: pointer;
                    font-size: 15px;
                    transition: transform 0.2s, box-shadow 0.2s;
                }
                
                .btn-select-all:hover {
                    transform: translateY(-2px);
                    box-shadow: 0 5px 15px rgba(216, 27, 96, 0.5);
                }
                
                .btn-deselect-all {
                    background: linear-gradient(135deg, #0288d1 0%, #01579b 100%);
                    color: white !important;
                    border: none;
                    padding: 12px 30px;
                    border-radius: 8px;
                    font-weight: 700 !important;
                    cursor: pointer;
                    font-size: 15px;
                    transition: transform 0.2s, box-shadow 0.2s;
                }
                
                .btn-deselect-all:hover {
                    transform: translateY(-2px);
                    box-shadow: 0 5px 15px rgba(2, 136, 209, 0.5);
                }
                
                .empty-state {
                    text-align: center;
                    padding: 50px 20px;
                    color: #000 !important;
                    font-weight: 700 !important;
                }
                
                .empty-state-icon {
                    font-size: 64px;
                    margin-bottom: 15px;
                    opacity: 0.5;
                }
                
                .info-box {
                    background: linear-gradient(135deg, #e0f7fa 0%, #b2ebf2 100%);
                    border-right: 5px solid #00acc1;
                    padding: 20px;
                    border-radius: 8px;
                    margin-bottom: 20px;
                }
                
                .info-box h3 {
                    margin: 0 0 15px 0;
                    color: #000 !important;
                    font-weight: 700 !important;
                }
                
                .info-box ul {
                    margin: 10px 0;
                    padding-right: 20px;
                }
                
                .info-box li {
                    margin: 8px 0;
                    color: #000 !important;
                    font-weight: 700 !important;
                }
                
                .info-box p {
                    color: #000 !important;
                    font-weight: 700 !important;
                }
                
                .support-links {
                    display: flex;
                    gap: 12px;
                    margin-bottom: 20px;
                    flex-wrap: wrap;
                }
                
                .btn-support {
                    background: linear-gradient(135deg, #d81b60 0%, #c2185b 100%);
                    color: white !important;
                    border: none;
                    padding: 12px 25px;
                    border-radius: 8px;
                    font-weight: 700 !important;
                    cursor: pointer;
                    text-decoration: none;
                    display: inline-block;
                    transition: transform 0.2s, box-shadow 0.2s;
                }
                
                .btn-support:hover {
                    transform: translateY(-2px);
                    box-shadow: 0 5px 15px rgba(216, 27, 96, 0.5);
                    color: white !important;
                }
                
                .btn-scan {
                    background: linear-gradient(135deg, #0288d1 0%, #01579b 100%);
                    color: white !important;
                    border: none;
                    padding: 12px 25px;
                    border-radius: 8px;
                    font-weight: 700 !important;
                    cursor: pointer;
                    text-decoration: none;
                    display: inline-block;
                    transition: transform 0.2s, box-shadow 0.2s;
                }
                
                .btn-scan:hover {
                    transform: translateY(-2px);
                    box-shadow: 0 5px 15px rgba(2, 136, 209, 0.5);
                    color: white !important;
                }
                
                .suspicious-table {
                    width: 100%;
                    border-collapse: collapse;
                }
                
                .suspicious-table thead {
                    background: linear-gradient(135deg, #d81b60 0%, #c2185b 100%);
                }
                
                .suspicious-table thead th {
                    color: white !important;
                    padding: 15px;
                    text-align: right;
                    font-weight: 700 !important;
                }
                
                .suspicious-table tbody tr {
                    border-bottom: 1px solid #f0f0f0;
                    transition: background-color 0.2s;
                }
                
                .suspicious-table tbody tr:hover {
                    background-color: #fff3e0;
                }
                
                .suspicious-table tbody td {
                    padding: 15px;
                    text-align: right;
                    color: #000 !important;
                    font-weight: 700 !important;
                }
                
                /* Toast Notification */
                .wp-smart-limit-toast {
                    position: fixed;
                    bottom: 30px;
                    left: 50%;
                    transform: translateX(-50%);
                    background: white;
                    color: #000;
                    padding: 16px 24px;
                    border-radius: 8px;
                    box-shadow: 0 5px 25px rgba(0,0,0,0.3);
                    z-index: 999999;
                    font-weight: 700;
                    opacity: 0;
                    transition: opacity 0.3s ease-in-out;
                }
                
                .wp-smart-limit-toast.show {
                    opacity: 1;
                }
                
                .wp-smart-limit-toast.info {
                    border-right: 5px solid #2196F3;
                }
                
                .wp-smart-limit-toast.success {
                    border-right: 5px solid #4CAF50;
                }
                
                .wp-smart-limit-toast.error {
                    border-right: 5px solid #f44336;
                }
                
                /* Loading Spinner */
                .wp-smart-limit-loading {
                    text-align: center;
                    padding: 40px;
                    color: #667eea !important;
                    font-weight: 700 !important;
                }
                
                @media (max-width: 768px) {
                    .hosts-table, .suspicious-table {
                        font-size: 13px;
                    }
                    
                    .hosts-table thead th,
                    .hosts-table tbody td,
                    .suspicious-table thead th,
                    .suspicious-table tbody td {
                        padding: 10px;
                    }
                    
                    .wp-smart-limit-stats {
                        grid-template-columns: 1fr;
                    }
                    
                    .action-buttons, .support-links {
                        flex-direction: column;
                    }
                    
                    .action-buttons button, .support-links a {
                        width: 100%;
                    }
                }
            </style>

            <div class="wp-smart-limit-header">
                <h1>🛡️ بلاک هوشمند وردپرس</h1>
                <p>مدیریت هوشمند اتصالات خارجی و امنیت پیشرفته سایت</p>
            </div>

            <div class="wp-smart-limit-stats">
                <div class="stat-card">
                    <div class="stat-number"><?php echo count($logged_hosts); ?></div>
                    <div class="stat-label">دامنه شناسایی شده</div>
                </div>
                <div class="stat-card">
                    <div class="stat-number"><?php echo count($blocked_hosts); ?></div>
                    <div class="stat-label">دامنه بلاک شده</div>
                </div>
                <div class="stat-card">
                    <div class="stat-number"><?php echo count($suspicious_hosts); ?></div>
                    <div class="stat-label">دامنه مشکوک</div>
                </div>
            </div>

            <h2 class="nav-tab-wrapper">
                <a href="#" class="nav-tab wp-smart-limit-tab <?php echo $tab == 'block' ? 'nav-tab-active' : ''; ?>" data-tab="block">
                    مدیریت بلاک
                </a>
                <a href="#" class="nav-tab wp-smart-limit-tab <?php echo $tab == 'suspicious' ? 'nav-tab-active' : ''; ?>" data-tab="suspicious">
                    اتصال‌های مشکوک
                </a>
            </h2>

            <div id="wp-smart-limit-tab-content">
                <?php $this->render_tab_content($tab); ?>
            </div>
        </div>

        <script>
        var wpSmartLimitCurrentTab = '<?php echo $tab; ?>';
        
        function wpSmartLimitShowToast(message, type) {
            type = type || 'info';
            var toast = jQuery('<div class="wp-smart-limit-toast ' + type + '">' + message + '</div>');
            jQuery('body').append(toast);
            
            setTimeout(function() {
                toast.addClass('show');
            }, 100);
            
            setTimeout(function() {
                toast.removeClass('show');
                setTimeout(function() {
                    toast.remove();
                }, 300);
            }, 4000);
        }
        
        jQuery(document).ready(function($) {
            // تب‌های Ajax
            $('.wp-smart-limit-tab').on('click', function(e) {
                e.preventDefault();
                var tab = $(this).data('tab');
                
                if (tab === wpSmartLimitCurrentTab) {
                    return;
                }
                
                $('.wp-smart-limit-tab').removeClass('nav-tab-active');
                $(this).addClass('nav-tab-active');
                
                $('#wp-smart-limit-tab-content').html('<div class="wp-smart-limit-loading">در حال بارگذاری...</div>');
                
                $.post(ajaxurl, {
                    action: 'wp_smart_limit_load_tab',
                    tab: tab,
                    nonce: '<?php echo wp_create_nonce('wp_smart_limit_tab'); ?>'
                })
                .done(function(response) {
                    if (response.success) {
                        $('#wp-smart-limit-tab-content').html(response.data.html);
                        wpSmartLimitCurrentTab = tab;
                        wpSmartLimitInitHandlers();
                    }
                })
                .fail(function() {
                    wpSmartLimitShowToast('خطا در بارگذاری تب', 'error');
                });
            });
            
            wpSmartLimitInitHandlers();
        });
        
        function wpSmartLimitInitHandlers() {
            var $ = jQuery;
            
            // دکمه عملیات سریع
            $('.wp-smart-limit-quick-action').off('click').on('click', function() {
                var button = $(this);
                var host = button.data('host');
                var action = button.data('action');
                var originalText = button.text();

                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'wp_smart_limit_quick_action',
                        nonce: '<?php echo wp_create_nonce('wp_smart_limit_quick_action'); ?>',
                        action_type: action,
                        host: host
                    },
                    beforeSend: function() {
                        button.prop('disabled', true).text('در حال پردازش...');
                    },
                    success: function(response) {
                        if (response.success) {
                            var newAction = (action === 'block') ? 'unblock' : 'block';
                            var newText = (action === 'block') ? 'رفع بلاک' : 'بلاک کردن';

                            button
                                .data('action', newAction)
                                .text(newText)
                                .prop('disabled', false)
                                .removeClass('btn-block btn-unblock')
                                .addClass(action === 'block' ? 'btn-unblock' : 'btn-block');

                            var checkbox = $('input[name="blocked[' + host + ']"]');
                            checkbox.prop('checked', action === 'block');

                            var statusCell = button.closest('tr').find('td:nth-child(3)');
                            if (action === 'block') {
                                statusCell.html('<span class="badge badge-blocked">بلاک شده</span>');
                            } else {
                                statusCell.html('<span class="badge badge-active">فعال</span>');
                            }
                            
                            wpSmartLimitShowToast('عملیات با موفقیت انجام شد', 'success');
                        } else {
                            button.prop('disabled', false).text(originalText);
                            wpSmartLimitShowToast('خطا در انجام عملیات', 'error');
                        }
                    },
                    error: function() {
                        button.prop('disabled', false).text(originalText);
                        wpSmartLimitShowToast('خطا در ارتباط با سرور', 'error');
                    }
                });
            });

            // دکمه انتخاب همه
            $('.wp-smart-limit-select-all').off('click').on('click', function() {
                $('input[name^="blocked["]').prop('checked', true);
            });

            // دکمه لغو انتخاب همه
            $('.wp-smart-limit-deselect-all').off('click').on('click', function() {
                $('input[name^="blocked["]').prop('checked', false);
            });
        }
        </script>
        <?php
    }

    private function render_tab_content($tab) {
        $logged_hosts = $this->get_logged_hosts();
        $blocked_hosts = get_option('wp_smart_limit_blocked_hosts', []);
        $suspicious_hosts = $this->detect_suspicious_hosts();
        $hosts_details = $this->get_hosts_with_details();
        
        if ($tab == 'block'):
        ?>
            <!-- بخش افزودن دامنه دستی -->
            <div class="wp-smart-limit-section">
                <h2 class="section-title">افزودن دامنه جدید</h2>
                <form method="post" class="add-host-form">
                    <?php wp_nonce_field('wp_smart_limit_save_hosts', '_wpnonce'); ?>
                    <input type="text" 
                           name="custom_host" 
                           placeholder="مثال: example.com یا *.example.com"
                           dir="ltr">
                    <button type="submit" name="add_custom_host" class="btn-primary">
                        افزودن به لیست بلاک
                    </button>
                </form>
                <p style="color: #000; font-size: 13px; margin: 10px 0 0 0; font-weight: 700;">
                    💡 می‌توانید از الگوهای Wildcard استفاده کنید (مثال: *.adservice.com)
                </p>
            </div>

            <!-- جدول دامنه‌های شناسایی شده -->
            <div class="wp-smart-limit-section">
                <h2 class="section-title">لیست دامنه‌های شناسایی شده</h2>
                
                <?php if (empty($logged_hosts)): ?>
                    <div class="empty-state">
                        <div class="empty-state-icon">📭</div>
                        <p>هنوز هیچ دامنه‌ای شناسایی نشده است</p>
                    </div>
                <?php else: ?>
                    <form method="post">
                        <?php wp_nonce_field('wp_smart_limit_save_hosts', '_wpnonce'); ?>
                        
                        <!-- تنظیمات به‌روزرسانی خودکار -->
                        <div style="background: #f8f9fa; padding: 20px; border-radius: 8px; margin-bottom: 25px; border: 2px solid #dee2e6;">
                            <h3 style="margin: 0 0 15px 0; color: #000; font-weight: 700; font-size: 16px;">
                                ⚙️ تنظیمات به‌روزرسانی خودکار وردپرس
                            </h3>
                            <p style="color: #000; font-weight: 700; margin: 0 0 15px 0; font-size: 14px; line-height: 1.8;">
                                با غیرفعال کردن این گزینه، وردپرس دیگر به صورت خودکار هسته، افزونه‌ها، قالب‌ها و ترجمه‌ها را به‌روزرسانی نمی‌کند. 
                                شما کنترل کامل به‌روزرسانی‌ها را خواهید داشت.
                            </p>
                            <div style="display: flex; align-items: center; gap: 15px;">
                                <label style="color: #000; font-weight: 700; font-size: 15px; min-width: 180px;">
                                    🔒 وضعیت به‌روزرسانی خودکار:
                                </label>
                                <select name="disable_auto_update" 
                                        style="padding: 10px 15px; font-size: 14px; font-weight: 700; border: 2px solid #ced4da; border-radius: 6px; background: white; color: #000; cursor: pointer; min-width: 150px;">
                                    <option value="0" <?php echo !get_option('wp_smart_limit_disable_auto_update', false) ? 'selected' : ''; ?>>
                                        ✅ فعال (پیش‌فرض)
                                    </option>
                                    <option value="1" <?php echo get_option('wp_smart_limit_disable_auto_update', false) ? 'selected' : ''; ?>>
                                        ❌ غیرفعال
                                    </option>
                                </select>
                            </div>
                        </div>
                        
                        <table class="hosts-table">
                            <thead>
                                <tr>
                                    <th style="width: 50px;">بلاک</th>
                                    <th>نام دامنه</th>
                                    <th style="width: 120px;">تعداد درخواست</th>
                                    <th style="width: 150px;">وضعیت</th>
                                    <th style="width: 150px;">عملیات سریع</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($hosts_details as $host_data): 
                                    if ($this->is_internal_host($host_data['host'])) continue;
                                    
                                    $is_blocked = in_array($host_data['host'], $blocked_hosts);
                                    $is_suspicious = in_array($host_data['host'], $suspicious_hosts);
                                ?>
                                <tr data-host="<?php echo esc_attr($host_data['host']); ?>">
                                    <td>
                                        <div class="checkbox-wrapper">
                                            <input type="checkbox" 
                                                   name="blocked[<?php echo esc_attr($host_data['host']); ?>]" 
                                                   <?php checked($is_blocked); ?>>
                                        </div>
                                    </td>
                                    <td>
                                        <span class="host-url"><?php echo esc_html($host_data['host']); ?></span>
                                        <?php if ($is_suspicious): ?>
                                            <span style="color: #ef6c00; margin-right: 8px;">⚠️</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <span class="host-count"><?php echo $host_data['count']; ?> بار</span>
                                    </td>
                                    <td>
                                        <?php if ($is_blocked): ?>
                                            <span class="badge badge-blocked">بلاک شده</span>
                                        <?php elseif ($is_suspicious): ?>
                                            <span class="badge badge-suspicious">مشکوک</span>
                                        <?php else: ?>
                                            <span class="badge badge-active">فعال</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <div class="quick-actions">
                                            <?php if ($is_blocked): ?>
                                                <button type="button" 
                                                        class="quick-action-btn btn-unblock wp-smart-limit-quick-action"
                                                        data-action="unblock"
                                                        data-host="<?php echo esc_attr($host_data['host']); ?>">
                                                    رفع بلاک
                                                </button>
                                            <?php else: ?>
                                                <button type="button" 
                                                        class="quick-action-btn btn-block wp-smart-limit-quick-action"
                                                        data-action="block"
                                                        data-host="<?php echo esc_attr($host_data['host']); ?>">
                                                    بلاک کردن
                                                </button>
                                            <?php endif; ?>
                                        </div>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                        
                        <div class="action-buttons">
                            <button type="submit" name="save_blocked_hosts" class="btn-save">
                                ذخیره تغییرات
                            </button>
                            <button type="button" class="btn-select-all wp-smart-limit-select-all">
                                انتخاب همه
                            </button>
                            <button type="button" class="btn-deselect-all wp-smart-limit-deselect-all">
                                لغو انتخاب همه
                            </button>
                        </div>
                    </form>
                <?php endif; ?>
            </div>

        <?php elseif ($tab == 'suspicious'): ?>
            <div class="wp-smart-limit-section">
                <h2 class="section-title">اتصال‌های مشکوک</h2>
                
                <div class="support-links">
                    <a href="https://mrbarati.com/forms/" target="_blank" class="btn-support">
                        پشتیبانی سایت (09153050096)
                    </a>
                    <a href="?page=smart-block&tab=suspicious" class="btn-scan">
                        اسکان مجدد
                    </a>
                </div>
                
                <div class="info-box">
                    <h3>📚 راهنمای افزونه بلاک هوشمند وردپرس</h3>
                    <p>این افزونه به شما کمک می‌کند تا اتصالات خارجی سایت وردپرسی خود را مدیریت کنید:</p>
                    <ul>
                        <li>تشخیص خودکار دامنه‌های مشکوک با بیش از <?php echo $this->suspicious_threshold; ?> اتصال</li>
                        <li>امکان بلاک/آزادسازی سریع دامنه‌ها با یک کلیک</li>
                        <li>پشتیبانی از الگوهای Wildcard برای بلاک دامنه‌های مشابه</li>
                        <li>نمایش وضعیت اتصالات در جداول جذاب</li>
                        <li>مدیریت کش برای عملکرد بهینه</li>
                        <li>لاگ خودکار تمام درخواست‌های خارجی</li>
                        <li>رهگیری منبع درخواست و نمایش تعداد اتصالات</li>
                        <li>نویسنده افزونه : براتی طراح و پشتیبان سایت های وردپرسی</li>
                    </ul>
                    <p><strong>برای پشتیبانی بیشتر:</strong> تماس با مهندس براتی شماره 09153050096</p>
                </div>
                
                <?php if (!empty($suspicious_hosts)): ?>
                    <table class="suspicious-table">
                        <thead>
                            <tr>
                                <th>ردیف</th>
                                <th>دامنه مشکوک</th>
                                <th>تعداد اتصال</th>
                                <th>علت مشکوک بودن</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php 
                            $counter = 1;
                            $hosts_details = $this->get_hosts_with_details();
                            foreach ($suspicious_hosts as $host):
                                $host_detail = array_filter($hosts_details, function($h) use ($host) {
                                    return $h['host'] === $host;
                                });
                                $host_detail = reset($host_detail);
                            ?>
                            <tr>
                                <td><?php echo $counter++; ?></td>
                                <td>
                                    <span class="host-url"><?php echo esc_html($host); ?></span>
                                </td>
                                <td>
                                    <?php echo $host_detail ? $host_detail['count'] : 0; ?> بار
                                </td>
                                <td>
                                    <span class="badge badge-suspicious">
                                        بیش از <?php echo $this->suspicious_threshold; ?> اتصال
                                    </span>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php else: ?>
                    <div class="empty-state">
                        <div class="empty-state-icon">✅</div>
                        <p><strong>هیچ اتصال مشکوکی یافت نشد!</strong></p>
                        <p style="color: #000;">سایت شما از نظر اتصالات خارجی در وضعیت مناسبی قرار دارد.</p>
                    </div>
                <?php endif; ?>
            </div>
        <?php
        endif;
    }

    public function handle_load_tab() {
        check_ajax_referer('wp_smart_limit_tab', 'nonce');
        
        $tab = sanitize_key($_POST['tab']);
        
        ob_start();
        $this->render_tab_content($tab);
        $html = ob_get_clean();
        
        wp_send_json_success(['html' => $html]);
    }

    private function handle_form_submissions() {
        if (isset($_POST['save_blocked_hosts']) && check_admin_referer('wp_smart_limit_save_hosts', '_wpnonce')) {
            $this->save_blocked_hosts();
            
            // ذخیره وضعیت غیرفعال کردن به‌روزرسانی خودکار
            $disable_auto_update = isset($_POST['disable_auto_update']) ? intval($_POST['disable_auto_update']) : 0;
            update_option('wp_smart_limit_disable_auto_update', $disable_auto_update);
        }

        if (isset($_POST['add_custom_host']) && check_admin_referer('wp_smart_limit_save_hosts', '_wpnonce')) {
            $this->add_custom_host();
        }
    }

    public function handle_http_request($preempt, $parsed_args, $url) {
        $host = parse_url($url, PHP_URL_HOST);
        
        if (!$host || $this->is_internal_host($host)) {
            return $preempt;
        }

        // بررسی بلاک بودن قبل از لاگ - جلوگیری از ارسال درخواست
        if ($this->is_blocked($host)) {
            $this->increment_block_counter();
            // برگرداندن WP_Error مانع ارسال درخواست HTTP می‌شود
            return new WP_Error(
                'http_request_blocked',
                sprintf('درخواست به %s توسط بلاک هوشمند بلاک شد', $host),
                array('blocked_by' => 'wp_smart_limit', 'host' => $host)
            );
        }

        // بررسی دامنه جدید برای نمایش Toast
        $existing_hosts = $this->get_logged_hosts();
        $is_new = !in_array($host, $existing_hosts);
        
        // لاگ کردن فقط درخواست‌های مجاز
        $this->log_host($host);

        // اگر دامنه جدید است، به لیست موقت اضافه کن
        if ($is_new) {
            $new_domains = get_transient('wp_smart_limit_new_domains') ?: [];
            if (!in_array($host, $new_domains)) {
                $new_domains[] = $host;
                set_transient('wp_smart_limit_new_domains', $new_domains, 300);
            }
        }

        return $preempt;
    }
    
    /**
     * بلاک کامل transient های به‌روزرسانی
     */
    public function block_update_transients($transient) {
        // اگر کاربر گزینه غیرفعال کردن به‌روزرسانی را فعال کرده
        if (get_option('wp_smart_limit_disable_auto_update', false)) {
            return (object) array(
                'last_checked' => time(),
                'updates' => array(),
                'no_update' => array()
            );
        }
        
        // اگر api.wordpress.org بلاک است، هیچ اطلاعات به‌روزرسانی برنگردان
        if ($this->is_blocked('api.wordpress.org')) {
            return (object) array(
                'last_checked' => time(),
                'updates' => array(),
                'no_update' => array()
            );
        }
        return $transient;
    }
    
    /**
     * چک کردن external بودن host
     */
    public function check_external_host($is_external, $host, $url) {
        if ($this->is_blocked($host)) {
            // اگر بلاک است، به عنوان internal برگردان تا درخواست ارسال نشود
            return false;
        }
        return $is_external;
    }

    public function handle_quick_action() {
        check_ajax_referer('wp_smart_limit_quick_action', 'nonce');
        
        $action = sanitize_key($_POST['action_type']);
        $host = sanitize_text_field($_POST['host']);
        $blocked = get_option('wp_smart_limit_blocked_hosts', []);

        if ('block' === $action) {
            if (!in_array($host, $blocked)) {
                $blocked[] = $host;
            }
        } elseif ('unblock' === $action) {
            $blocked = array_diff($blocked, [$host]);
        }
        
        update_option('wp_smart_limit_blocked_hosts', array_values($blocked));
        $this->clear_cache();
        wp_send_json_success(['action' => $action, 'host' => $host]);
    }

    public function show_blocked_count() {
        $count = get_transient('wp_smart_limit_blocked_count');
        if ($count > 0) {
            ?>
            <div class="notice notice-warning is-dismissible">
                <p style="color: #000; font-weight: 700;">
                    <strong>بلاک هوشمند وردپرس:</strong>
                    تعداد <?php echo (int)$count; ?> اتصال خارجی بلاک شد.
                </p>
            </div>
            <?php
            delete_transient('wp_smart_limit_blocked_count');
        }
    }

    private function get_blocked_hosts() {
        $blocked = wp_cache_get($this->cache_key, 'options');
        if (false === $blocked) {
            $blocked = get_option('wp_smart_limit_blocked_hosts', []);
            wp_cache_set($this->cache_key, $blocked, 'options', 3600);
        }
        return $blocked;
    }

    private function get_logged_hosts() {
        $logged = get_option('wp_smart_limit_logged_hosts', []);
        $hosts_counts = array_count_values($logged);
        arsort($hosts_counts);
        return array_keys($hosts_counts);
    }
    
    private function get_hosts_with_details() {
        $logged = get_option('wp_smart_limit_logged_hosts', []);
        $hosts_counts = array_count_values($logged);
        arsort($hosts_counts);
        
        $result = [];
        foreach ($hosts_counts as $host => $count) {
            $result[] = [
                'host' => $host,
                'count' => $count
            ];
        }
        return $result;
    }
    
    private function detect_suspicious_hosts() {
        $logs = get_option('wp_smart_limit_logged_hosts', []);
        if(empty($logs)) return [];
        
        $frequency = array_count_values($logs);
        
        return array_keys(array_filter($frequency, function($count, $host) {
            return $count > $this->suspicious_threshold && !$this->is_whitelisted($host);
        }, ARRAY_FILTER_USE_BOTH));
    }

    private function is_blocked($host) {
        $blocked_hosts = $this->get_blocked_hosts();
        foreach ($blocked_hosts as $pattern) {
            if ($this->match_wildcard($host, $pattern)) {
                return true;
            }
        }
        return false;
    }
    
    private function match_wildcard($host, $pattern) {
        $pattern = preg_quote($pattern, '/');
        $pattern = str_replace('\*', '.*', $pattern);
        return (bool) preg_match("/^{$pattern}$/i", $host);
    }

    private function is_internal_host($host) {
        $site_host = parse_url(home_url(), PHP_URL_HOST);
        return strtolower($site_host) === strtolower($host);
    }

    private function is_whitelisted($host) {
        $whitelist = apply_filters('wp_smart_limit_whitelisted_hosts', [
            '*.wordpress.org',
            '*.google.com',
            '*.googleapis.com',
            '*.gstatic.com',
            'api.jetpack.com'
        ]);
        foreach ($whitelist as $pattern) {
            if ($this->match_wildcard($host, $pattern)) {
                return true;
            }
        }
        return false;
    }

    private function log_host($host) {
        $logged = get_option('wp_smart_limit_logged_hosts', []);
        $logged[] = $host;
        if (count($logged) > 500) {
            $logged = array_slice($logged, -300);
        }
        update_option('wp_smart_limit_logged_hosts', $logged, 'no');
    }

    private function increment_block_counter() {
        $count = (int) get_transient('wp_smart_limit_blocked_count');
        set_transient('wp_smart_limit_blocked_count', $count + 1, 3600);
    }

    private function save_blocked_hosts() {
        $blocked_posted = isset($_POST['blocked']) ? array_keys($_POST['blocked']) : [];
        $blocked = array_map('sanitize_text_field', $blocked_posted);
        update_option('wp_smart_limit_blocked_hosts', $blocked);
        $this->clear_cache();
        add_settings_error('wp_smart_limit', 'saved', 'تنظیمات با موفقیت ذخیره شد!', 'updated');
    }

    private function add_custom_host() {
        if (!empty($_POST['custom_host'])) {
            $host = sanitize_text_field(trim($_POST['custom_host']));
            
            $logged = get_option('wp_smart_limit_logged_hosts', []);
            if (!in_array($host, $logged)) {
                $logged[] = $host;
                update_option('wp_smart_limit_logged_hosts', $logged);
            }
            
            $blocked = get_option('wp_smart_limit_blocked_hosts', []);
            if (!in_array($host, $blocked)) {
                $blocked[] = $host;
                update_option('wp_smart_limit_blocked_hosts', $blocked);
                $this->clear_cache();
                add_settings_error('wp_smart_limit', 'host_added', 'دامنه ' . esc_html($host) . ' با موفقیت به لیست بلاک اضافه شد!', 'updated');
            } else {
                add_settings_error('wp_smart_limit', 'host_exists', 'این دامنه قبلاً به لیست اضافه شده است!', 'error');
            }
        }
    }

    private function clear_cache() {
        wp_cache_delete($this->cache_key, 'options');
    }
}

// راه‌اندازی افزونه
add_action('plugins_loaded', function() {
    WP_Smart_Limit::get_instance();
});

// پشتیبانی از WP-CLI
if (defined('WP_CLI') && WP_CLI) {
    class WP_Smart_Limit_CLI {
        public function list($args, $assoc_args) {
            $logged = get_option('wp_smart_limit_logged_hosts', []);
            $blocked = get_option('wp_smart_limit_blocked_hosts', []);
            
            WP_CLI::line('دامنه‌های شناسایی شده:');
            foreach (array_unique($logged) as $host) {
                $status = in_array($host, $blocked) ? '🚫 بلاک' : '✓ فعال';
                WP_CLI::line("  $status - $host");
            }
        }

        public function block($args, $assoc_args) {
            $host = $args[0];
            $blocked = get_option('wp_smart_limit_blocked_hosts', []);
            
            if (!in_array($host, $blocked)) {
                $blocked[] = $host;
                update_option('wp_smart_limit_blocked_hosts', $blocked);
                WP_CLI::success("دامنه $host بلاک شد.");
            } else {
                WP_CLI::warning("دامنه $host قبلاً بلاک شده است.");
            }
        }

        public function unblock($args, $assoc_args) {
            $host = $args[0];
            $blocked = get_option('wp_smart_limit_blocked_hosts', []);
            
            if (in_array($host, $blocked)) {
                $blocked = array_diff($blocked, [$host]);
                update_option('wp_smart_limit_blocked_hosts', array_values($blocked));
                WP_CLI::success("بلاک دامنه $host رفع شد.");
            } else {
                WP_CLI::warning("دامنه $host بلاک نشده است.");
            }
        }

        public function clear($args, $assoc_args) {
            delete_option('wp_smart_limit_logged_hosts');
            delete_option('wp_smart_limit_blocked_hosts');
            WP_CLI::success('تمام داده‌ها پاک شد.');
        }
    }

    WP_CLI::add_command('smart-limit', 'WP_Smart_Limit_CLI');
}

// ============================================================
// 🔥 بخش پاکسازی کامل دیتابیس هنگام حذف افزونه
// ============================================================

/**
 * تابع پاکسازی کامل - اجرا میشه وقتی افزونه DELETE شد (نه deactivate)
 */
register_uninstall_hook(__FILE__, 'wp_smart_limit_complete_uninstall');

function wp_smart_limit_complete_uninstall() {
    global $wpdb;
    
    // 1. حذف تمام option ها
    $option_patterns = [
        'wp_smart_limit%',
        'smart_limit%',
        '%_smart_limit%',
    ];
    
    foreach ($option_patterns as $pattern) {
        $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM {$wpdb->options} WHERE option_name LIKE %s",
                $pattern
            )
        );
    }
    
    // 2. حذف تمام transient ها
    $transient_patterns = [
        '_transient_%smart_limit%',
        '_transient_timeout_%smart_limit%',
        '_site_transient_%smart_limit%',
        '_site_transient_timeout_%smart_limit%',
    ];
    
    foreach ($transient_patterns as $pattern) {
        $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM {$wpdb->options} WHERE option_name LIKE %s",
                $pattern
            )
        );
    }
    
    // 3. حذف از postmeta (اگه داشته باشه)
    $wpdb->query(
        "DELETE FROM {$wpdb->postmeta} 
         WHERE meta_key LIKE '%smart_limit%' 
            OR meta_key LIKE '%wp_smart_limit%'"
    );
    
    // 4. حذف از usermeta (اگه داشته باشه)
    $wpdb->query(
        "DELETE FROM {$wpdb->usermeta} 
         WHERE meta_key LIKE '%smart_limit%' 
            OR meta_key LIKE '%wp_smart_limit%'"
    );
    
    // 5. پاک کردن کش
    wp_cache_flush();
    
    // 6. حذف کش object
    wp_cache_delete('wp_smart_limit_blocked_hosts_cache', 'options');
    
    // 7. پاک کردن تمام option های مرتبط با افزونه (دور دوم برای اطمینان)
    delete_option('wp_smart_limit_settings');
    delete_option('wp_smart_limit_blocked_hosts');
    delete_option('wp_smart_limit_logged_hosts');
    delete_option('wp_smart_limit_blocked_hosts_cache');
    delete_option('wp_smart_limit_version');
    delete_option('wp_smart_limit_options');
    
    delete_transient('wp_smart_limit_blocked_count');
    delete_transient('wp_smart_limit_new_domains');
    
    // 8. حذف site transient ها
    delete_site_transient('wp_smart_limit_update_check');
}